/*
 * AUTHOR: Gonçalo Morais (gnrm@fct.unl.pt) 
 *
 * Version :: 0.0.07 
 * STATE :: Highly Unstable.
 *
 * NUMDE is a scientific
 * library and pretends to implement the state-of-art of the
 * algorithms related with Differential Equations.  Although this
 * project is mainly written by one person, it received the valuable
 * contribution of several people. See the home page of the project
 * for further details.
 */

/*
 *  This file is part of NUMDE.
 *
 *  NUMDE is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  NUMDE is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with NUMDE.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef __NUMDE_DYNAMICS_C__
#define __NUMDE_DYNAMICS_C__


#include "numde.h"


/********************************************************************
 ********************************************************************
 ********************************************************************
 *
 * The next set of functions are the implementation of RUNGE-KUTTA of
 * order 4. The implementation of this algorithm was made independent
 * from the dimension of the problem.
 *
 ********************************************************************
 ********************************************************************
 *******************************************************************/

 
/* In the computation of the Runge-Kutta algorithm of fouth order, is
   necessary to use for vectors k1-k4, to save the values of the values
   of the vector field in 4 different points. As usual, this is simply an
   array of doubles, that is indeed an array of vectors of a certain
   dimension. The next macro allow us to refer to the beginning of each
   of these vectors. You should also note that the implementation
   reserves not 4 but 5 of these vectors. This is so simply because the
   first vector KK(0) is the place where the values are temporarily
   preserved (see macro RK_SUM bellow.)
*/ 
#define KK(i) (kk+i*dimension)  


/* In Runge Kutta algorithm of fourth order, in problems of dimension
   bigger than one, the algorithm computes the values of auxiliary
   vectors k1-k4 in a recurrent fashion. However we must preserve the
   original value of each of these vectors.

   To keep everything correct, we allocate one extra vector k0, and
   use it to carry on all the calculations. In this way, we preserve
   everything that we need. 

   The macro RK_SUM(i,dt) just defines a way to say that to the
   original vector "initial", we add dt*ki in the usual way.
*/
#define RK_SUM(i,dt)                  \
  do {                                \
  unsigned ii;                        \
  double *a, *b;                      \
  a = KK(0);                          \
  b = KK(i);                          \
  for (ii=0 ; ii<dimension ;++ii) {   \
    a[ii] = initial[ii] + dt * b[ii]; \
  }                                   \
  } while(0)




/* Now a giant step for men, a small step for the mankind */
void ND_RungeKutta ( NDorbit *out , 
		     unsigned long position , 
		     double tj ,
		     double *kk ,
		     vfield function) 
{
  unsigned int dimension = out->dim ;

  double *initial, *result , htj = tj/2. ;
  initial = NDp( out, position) ;
  result = NDp( out , position+1 ) ;

  /* first set of the algorithm */
  function( KK(1) , dimension , initial ) ;
  RK_SUM(1, htj);

  function( KK(2) , dimension , KK(0)) ;
  RK_SUM(2, htj);

  function( KK(3) , dimension , KK(0)) ;
  RK_SUM(3, tj);

  function( KK(4) , dimension , KK(0)) ;

  unsigned i ;
  for ( i = 0 ; i < dimension ; i++ )
    result[i] = initial[i] + tj * (kk[i+dimension] + 2 *kk[i+2*dimension] 
				   + 2* kk[i+3*dimension] + kk[i+4*dimension])/6. ;
}		

NDorbit* run_RungeKutta ( unsigned long int nsteps ,
			   unsigned dimension ,
			   double *in ,
			   double tj ,
			   vfield function)
{
  
  NDorbit *out = NDorbit_alloc ( nsteps , dimension ) ;
  double *kk;

  kk = malloc ( 5 * dimension * sizeof(double)) ;

  memcpy(NDp(out,0), in , dimension * sizeof(double)) ;

  unsigned long int i ;
  for ( i = 0 ; i < nsteps-1 ; i++ ){
    fprintf( stdout , "Evaluating step %lu of %lu\r" , i+2 , nsteps ) ;
    ND_RungeKutta( out , i , tj , kk , function ) ;
  }

  free(kk) ;
  printf( "\r\n\n") ;

  fprintf( stdout , "The orbit was computed by the Runge-Kutta algorithm of 4th order of NUMDE library.\n" ) ;
  fprintf( stdout , "NUMDE is free software (as in freedom).\n" ) ;
  fprintf( stdout , "Please visit the homepage of the project (gnupost.com) for further information.\n" );
  fprintf( stdout , "For bug report or suggestions, please contact Goncalo Morais (gnrm@fct.unl.pt).\n\n" ) ;
  return out ;
}

/********************************************************************
 ********************************************************************
 ********************************************************************
 * The next section is the definition of several classical vector
 * fields. Note that most part of them are implemented in two
 * stages. First we define the most general, with a whole range set of
 * parameters. After we define, if this makes sense, the classical
 * case.
 ********************************************************************
 ********************************************************************
 ********************************************************************/

/* the first is the lorenz vfield. this can not be used in runge-kutta
   directely because has not the structure of a vfield*/
void Lorenz ( double *out , unsigned dimension , const double *in , double a , double b , double c )
{
  out[0] = a * ( in[1] - in[0] ) ;
  out[1] = b * in[0] - in[1] - in[0] * in[2] ;
  out[2] = c * in[2] + in[0] * in[1] ;
}
/* Now the classical values for Lorenz Attractor (to be used in runge-kutta). */
void CLorenz ( double *out , unsigned dimension , const double *in )
{
  Lorenz( out , dimension , in , 10 , 28 , -8/3. ) ;
}

/* Equally famous is the case of the Rossler attractor. */
void Rossler ( double *out , unsigned dimension , const double *in )
{
  out[0] = -in[1] -in[2] ;
  out[1] = in[0] + .25 * in[1] ;
  out[2] = 1 + in[2] *( in[0] - 5.5 ) ;
}
#endif 
