/*
 * AUTHOR: Gonçalo Morais (gnrm@fct.unl.pt) 
 *
 * Version :: 0.0.07 
 * STATE :: Highly Unstable.
 *
 * NUMDE is a scientific
 * library and pretends to implement the state-of-art of the
 * algorithms related with Differential Equations.  Although this
 * project is mainly written by one person, it received the valuable
 * contribution of several people. See the home page of the project
 * for further details.
 */

/*
 *  This file is part of NUMDE.
 *
 *  NUMDE is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  NUMDE is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with NUMDE.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef __NUMDE_C__
#define __NUMDE_C__


#include "numde.h"


/*****************************************************************************
 *****************************************************************************
 * When we are dealing with numerical algorithms to simulate a certain
 * behavior, it is good to produce a lot of points. However, we do not
 * need so many points when we want to represent the data, for
 * example, in a graphic. The next set of routines are used precisely
 * to discard some of the points.
 *****************************************************************************
 ****************************************************************************/
NDorbit* NDorbit_cut ( NDorbit *in , unsigned gap )
{
  if ( in == NULL )
    return NULL ;

  unsigned long newlength =  in->number / (gap + 0.0) - 1 ;
  unsigned dimension = in->dim ;
  NDorbit *out = NDorbit_alloc ( newlength , dimension ) ;

  unsigned long i ;
  unsigned j ;

  for ( i = 0 ; i < newlength ; i++ )
    for ( j = 0 ; j < dimension ; j++ )
      out->mem[i*dimension + j] = in->mem[i*dimension*gap + j] ;

  return out ;
  
}


/*****************************************************************************
 *****************************************************************************
 * We can induce a vector structure in a NDorbit, by create a NDorbit
 * of the difference between two consecutive points in the original
 * NDorbit. 
 *****************************************************************************
 ****************************************************************************/
NDorbit* NDorbit_diff ( NDorbit *in )
{
  NDorbit *out =NDorbit_alloc ( in->number - 1 , in->dim ) ; 
  
  unsigned long i , nvectors = in->number / in->dim ;
  unsigned j ;

  for ( i = 0 ; i < nvectors ; i++ )
    for ( j = 0 ; j < in->dim ; j++ )
      out->mem[i*in->dim + j] = in->mem[(i + 1)*in->dim + j] - in->mem[i*in->dim + j] ;

  return out ;
}

/* 
 * NDorbitCombinatoricSum
 * We do control the dimention of both orbits to see if they
 * match. Carefull must be taken in order to do not sum things that
 * may be quite absurd. We emit an error message and return NULL.
 *
 * This function just sum every "vector" in NDorbit *trans to every
 * "point" in NDorbit *ref creating a grid of points (no triangles or normals are created).
 *
 */

NDorbit* NDorbitCombinatoricSum ( NDorbit *ref , NDorbit *trans )
{

  if ( ref->dim != trans->dim ){
    printf("ERROR: You are trying to sum NDorbit's with different dimension.\n"
	   "The computation will not continue.\n" 
	   "Please read the documentation of NDorbit_sum in NUMDE library.\n");
    return NULL ;
  }
  
  NDorbit *out = NDorbit_alloc( ref->number * trans->number , ref->dim ) ; 
  
  unsigned long i , j ;
  unsigned k ;
  
  /*
   * There are several precautions that we had while writing the
   * following code.
   *
   * We had two options. The first was to compute the translations of
   * each vector of *ref by all vectors in *trans. The second was to
   * compute the translation of *ref by a single vector in *trans one
   * at the time. We prefered the second option. The reason is quite
   * obvious, thinking that we want to use this reasoning building a
   * mesh of the tube. Put all the vectors in the right order is very
   * important.
   * 
   * This way, imagine that *ref has 5 vectors, r0, ... ,r4 , and
   * *trans has 3 vectors, t0,t1,t2. Then *out will have 3x5 vectors,
   * that can be placed in a matrix by
   *
   * out = r0+t0 r1+t0 r2+t0 r3+t0 r4+t0
   *       r0+t1 r1+t1 r2+t1 r3+t1 r4+t1
   *       r0+t2 r1+t2 r2+t2 r3+t2 r4+t2
   *
   * Searching by rows in this matrix, the k-component of vector
   * out[i,j], that is the sum of vector ri with vector tj, will be
   * placed in out[(i+jx5)*dim + k]. This is the spirit of the
   * following (obscure) instruction. Classical, but always beautiful.
   *
   */

  for (j = 0 ; j < trans->number ; j++ )
    for ( i = 0 ; i < ref->number ; i++ )
      for ( k = 0 ; k < ref->dim ; k++ )
	out->mem[(i+ j * ref->number) * ref->dim + k] = ref->mem[i * ref->dim + k] + trans->mem[ j * trans->dim + k] ; 
  
  return out ;
}

NDtmesh * NDtmeshCombinatoricSum ( NDorbit *circle , NDorbit *centers )
{
  if ( circle->dim != centers->dim ){
    printf("ERROR: You are trying to create a mesh from structures with different dimensions.\n" );
    return NULL ;
  }

  unsigned triside = 3 ; /* the number of sides in a triangle! */

  NDtmesh *out = NDtmesh_alloc( centers->number , circle->number , circle->dim );  

  unsigned long i , j ;
  unsigned k ;
  
/*   The next instruction, that is the body of this function, follows 
 *   exactly the same spirit of the similar instruction in function 
 *   NDorbit_combinatoric_sum. Please read it carefully. 
 *
 *   In the next few lines we fill up the coordinates of the mesh.
 */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < circle->number ; i++ )
      for ( k = 0 ; k < circle->dim ; k++ )
	out->coord[( i + j * circle->number ) * circle->dim + k ] =
	  circle->mem[i * circle->dim + k] + centers->mem[j * centers->dim + k] ;

  
  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  return out ;
}


/* 
 *  The function NDcombinatoric_sum works perfectly well in situations
 *  where we want to create a tubular mesh for things with curvuture
 *  equal to 0. In all other cases we need more careful and we must add
 *  an information about how do things are curving.
 */


NDtmesh * NDctCombinatoricSum ( NDorbit *circle , NDorbit *centers )
{
  if ( circle->dim != centers->dim ){
    printf("ERROR: You are trying to create a mesh from structures with different dimensions.\n" );
    return NULL ;
  }

  unsigned triside = 3 ; /* the number of sides in a triangle! */

  NDtmesh *out = NDctmesh_alloc( centers->number , circle->number , circle->dim );  

  unsigned long i , j ;
  unsigned k ;
  
/*   The next instruction, that is the body of this function, follows 
 *   exactly the same spirit of the similar instruction in function 
 *   NDorbit_combinatoric_sum. Please read it carefully. 
 *
 *   In the next few lines we fill up the coordinates of the mesh.
 */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < circle->number ; i++ )
      for ( k = 0 ; k < circle->dim ; k++ )
	out->coord[( i + j * circle->number ) * circle->dim + k ] =
	  circle->mem[i * circle->dim + k] + centers->mem[j * centers->dim + k] ;

  
  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  /* now we will add more triangles to close the mesh */
  j = out->circ - 1 ;
  
  for ( k = 0 ; k < out->rad ; k++ )
    {
      if ( k < out->rad - 1 ) 
	{
	  out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 2] = k  ;
	  
	  out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 ;
	  out->tri[2 * triside * ( k + j * out->rad) + 5] = k ;
	}
    }

  return out ;
}


NDstmesh * NDcstCombinatoricSum ( NDorbit *circle , NDorbit *centers )
{
  if ( circle->dim != centers->dim ){
    printf("ERROR: You are trying to create a mesh from structures with different dimensions.\n" );
    return NULL ;
  }

  unsigned triside = 3 ; /* the number of sides in a triangle! */

  NDstmesh *out = NDcstmesh_alloc( centers->number , circle->number , circle->dim );  

  unsigned long i , j ;
  unsigned k ;
  
/*   The next instruction, that is the body of this function, follows 
 *   exactly the same spirit of the similar instruction in function 
 *   NDorbitCombinatoricSum. Please read it carefully. 
 *
 *   In the next few lines we fill up the coordinates of the mesh.
 */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < circle->number ; i++ )
      for ( k = 0 ; k < circle->dim ; k++ )
	out->coord[( i + j * circle->number ) * circle->dim + k ] =
	  circle->mem[i * circle->dim + k] + centers->mem[j * centers->dim + k] ;

  
  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  /* now we will add more triangles to close the mesh */
  j = out->circ - 1 ;
  
  for ( k = 0 ; k < out->rad ; k++ )
    {
      if ( k < out->rad - 1 ) 
	{
	  out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 2] = k  ;
	  
	  out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 ;
	  out->tri[2 * triside * ( k + j * out->rad) + 5] = k ;
	}
    }

  
  /* now the normal vectors */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < circle->number ; i++ )
      for ( k = 0 ; k < circle->dim ; k++ )
	out->nvec[( i + j * circle->number ) * circle->dim + k ] =
	  out->coord[( i + j * circle->number ) * circle->dim + k ] - centers->mem[j * centers->dim + k] ;

  return out ;
}




/* 
 * The following function (NDTestSum while it does not have a definite
   name) computes a tubular mesh around the points of the NDorbit
   *centers, perpendicular to the vector given at the same point, by
   the vector field */

NDstmesh * NDTestSum ( NDorbit *centers , 
		       NDorbit *radius, 
		       void (*vectorf)(double * , double * ))
{
  unsigned triside = 3 ; /* the number of sides in a triangle! */
  /* the mesh that we want to produce */
  NDstmesh *out = NDcstmesh_alloc( centers->number , radius->number , radius->dim );
  
  /* just a structure to record the transformation by EulerAngles of
     original *radius */
  NDorbit *trad = NDorbit_alloc( radius->number , radius->dim ) ; 
  
  /* the vectors that will be used to compute the values of the vector field */
  double *tmpin  = calloc( centers->dim , sizeof(double) ) ;
  double *tmpout = calloc( centers->dim , sizeof(double) ) ;


  unsigned long i , j ;
  unsigned k ;
  
  /*   The next instruction, that is the body of this function, follows 
   *   exactly the same spirit of the similar instruction in function 
   *   NDorbitCombinatoricSum. Please read it carefully. 
   *
   *   In the next few lines we fill up the coordinates of the mesh.
   */
  for ( j = 0 ; j < centers->number ; j++ ){
    for ( k = 0 ; k < centers->dim ; k++ ) 
      tmpin[k] = centers->mem[j * centers->dim + k] ;
    vectorf(tmpin,tmpout) ;
    EulerAngles( radius , tmpout , trad ) ;
    for ( i = 0 ; i < radius->number ; i++ )
      for ( k = 0 ; k < radius->dim ; k++ ) 
	out->coord[( i + j * radius->number ) * radius->dim + k ] =
	  trad->mem[i * radius->dim + k] + centers->mem[j * centers->dim + k] ;
  }

  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  /* now we will add more triangles to close the mesh */
  j = out->circ - 1 ;
  
  for ( k = 0 ; k < out->rad ; k++ )
    {
      if ( k < out->rad - 1 ) 
	{
	  out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 2] = k  ;
	  
	  out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 ;
	  out->tri[2 * triside * ( k + j * out->rad) + 5] = k ;
	}
    }

  
  /* now the normal vectors */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < radius->number ; i++ )
      for ( k = 0 ; k < radius->dim ; k++ )
	out->nvec[( i + j * radius->number ) * radius->dim + k ] =
	  out->coord[( i + j * radius->number ) * radius->dim + k ] - centers->mem[j * centers->dim + k] ;



  /* Like Tibet, we want free memory */
  NDorbit_free(trad) ;
  free(tmpin) ;
  free(tmpout) ;

  return out ;
}

/* This is exactly the same principle but generates a mesh from a
   vector field that is defined parametrically */
NDstmesh* NDTestSumP ( NDorbit *centers , 
			      NDorbit *radius, 
			      double *parameter ,
			      void (*vectorf)(double * , double ))
{
  unsigned triside = 3 ; /* the number of sides in a triangle! */
  /* the mesh that we want to produce */
  NDstmesh *out = NDcstmesh_alloc( centers->number , radius->number , radius->dim );
  
  /* just a structure to record the transformation by EulerAngles of
     original *radius */
  NDorbit *trad = NDorbit_alloc( radius->number , radius->dim ) ; 
  
  /* the vectors that will be used to compute the values of the vector field */
  double *tmpout = calloc( centers->dim , sizeof(double) ) ;


  unsigned long i , j ;
  unsigned k ;
  
  /*   The next instruction, that is the body of this function, follows 
   *   exactly the same spirit of the similar instruction in function 
   *   NDorbitCombinatoricSum. Please read it carefully. 
   *
   *   In the next few lines we fill up the coordinates of the mesh.
   */
  for ( j = 0 ; j < centers->number ; j++ ){
    vectorf(tmpout, parameter[j]) ;
    EulerAngles( radius , tmpout , trad ) ;
    for ( i = 0 ; i < radius->number ; i++ ){
      for ( k = 0 ; k < radius->dim ; k++ ) 
	out->coord[( i + j * radius->number ) * radius->dim + k ] =
	  trad->mem[i * radius->dim + k] + centers->mem[j * centers->dim + k] ;
    }
  }

  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  /* now we will add more triangles to close the mesh */
  j = out->circ - 1 ;
  
  for ( k = 0 ; k < out->rad ; k++ )
    {
      if ( k < out->rad - 1 ) 
	{
	  out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 2] = k  ;
	  
	  out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 ;
	  out->tri[2 * triside * ( k + j * out->rad) + 5] = k ;
	}
    }

  
  /* now the normal vectors */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < radius->number ; i++ )
      for ( k = 0 ; k < radius->dim ; k++ )
	out->nvec[( i + j * radius->number ) * radius->dim + k ] =
	  out->coord[( i + j * radius->number ) * radius->dim + k ] - centers->mem[j * centers->dim + k] ;



  /* Like Tibet, we want free memory */
  NDorbit_free(trad) ;
  free(tmpout) ;

  return out ;
}





  
#endif
