/*
 * AUTHOR: Gonçalo Morais (gnrm@fct.unl.pt) 
 *
 * Version :: 0.0.07 
 * STATE :: Highly Unstable.
 *
 * NUMDE is a scientific
 * library and pretends to implement the state-of-art of the
 * algorithms related with Differential Equations.  Although this
 * project is mainly written by one person, it received the valuable
 * contribution of several people. See the home page of the project
 * for further details.
 */

/*
 *  This file is part of NUMDE.
 *
 *  NUMDE is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  NUMDE is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with NUMDE.  If not, see <http://www.gnu.org/licenses/>.
 *
 */


#ifndef _NUMDE_H__
#define _NUMDE_H__

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <string.h>

/* An NDorbit is just an array of doubles. But this array has more
   structure, because is indeed an array of vectors with dimension
   dim. The next macro help us to point the beginning of each of these
   vectors. */
#define NDp(in,i) (in->mem+(i)*in->dim)    



/* 
 * The datatype rgb is used to produce the output for some specific
 * formats like POVRAY.
*/
typedef struct {
  double r ;
  double g ;
  double b ;
} Color ;

/* 
 * The datatype TDvector is used to position cameras and other vector
 * calculations of cameras in POVRAY.
*/
typedef struct{
  double x;
  double y;
  double z;
} TDvector;

/* For several kind of problems we need to deal with multidimensional
   structures, like in the case of N-body problem. This is the basic
   structure for that case. */

typedef struct {
  unsigned dim ;
  unsigned long int number ;
  double *mem ;
} NDorbit ;

/* the next typedef just creates a prototype for what is called a
   vector field. the arguments are two arrays of doubles, and an
   unsigned int. the first array is the place where the result will be
   printed, while the unsigned integer in the second argument,
   controls the dimension of the second array, that is the place where
   the vector field will be evaluated. */
typedef  void (*vfield)( double * , unsigned , const double *) ;


typedef struct {
  unsigned dim ;                /* the dimension of the space where mesh lives */
  unsigned rad ;                /* the number of points in each section */
  unsigned long circ ;          /* the number of sections (circles) */
  unsigned long nvert ;         /* the number of vertices */
  double *coord ;               /* where the coordinates of all this will be */
  unsigned long ntri ;          /* the number of triangles */
  unsigned long *tri ;          /* the indices of the vertex that belong to each triangle */
} NDtmesh ;

typedef struct {
  unsigned dim ;                /* the dimension of the space where mesh lives */
  unsigned rad ;                /* the number of points in each section */
  unsigned long circ ;          /* the number of sections (circles) */
  unsigned long nvert ;         /* the number of vertices */
  double *coord ;               /* where the coordinates of all this will be */
  unsigned long ntri ;          /* the number of triangles */
  unsigned long *tri ;          /* the indices of the vertex that belong to each triangle */
  unsigned long nnvec ;         /* the number of normal vectors */
  double *nvec ;                /* the coordinates of normal vectors */ 
} NDstmesh ;

typedef struct {
  unsigned rows;
  unsigned cols;
  double *mem;
} Matrix;


/******************************************************************************
 ******************************************************************************
 ************** Contents of numde_mem.c ***************************************
 ******************************************************************************
 *****************************************************************************/
NDorbit* NDorbit_alloc ( unsigned long int nelements , unsigned int dimension ) ;
void NDorbit_free ( NDorbit *in ) ;
/* things about meshes */
NDtmesh* NDtmesh_alloc ( unsigned long ncirc , unsigned nrad , unsigned dimension ) ;
NDtmesh* NDctmesh_alloc ( unsigned long ncirc , unsigned nrad , unsigned dimension ) ;
NDstmesh* NDcstmesh_alloc ( unsigned long ncirc , unsigned nrad , unsigned dimension ) ;
NDstmesh* stubular_NDmesh_alloc ( unsigned long ncirc , unsigned nrad , unsigned dimension ) ;

/* serves perfectly for open and closed mesh's */
void NDtmesh_free ( NDtmesh *in ) ;
void NDstmesh_free ( NDstmesh *in ) ;


/******************************************************************************
 ******************************************************************************
 ************** Contents of numde_print.c *************************************
 ******************************************************************************
 *****************************************************************************/
/* functions for structured print of NDorbits */
/* the implementation of these functions is in numde_print.c */
void NDorbit_print ( NDorbit *in ) ;
void NDorbit_fprint ( char *filename , NDorbit *in ) ;
void tubular_NDmesh_coord_print ( NDtmesh *mesh ) ;
void tubular_NDmesh_tri_print ( NDtmesh *mesh ) ;
void NDmeshCoordPrintST ( NDstmesh *mesh ) ;


/******************************************************************************
 ******************************************************************************
 ************** Contents of numde.c *******************************************
 ******************************************************************************
 *****************************************************************************/

NDorbit* NDorbit_diff ( NDorbit *in ) ;
NDorbit* NDorbit_cut ( NDorbit *in , unsigned gap ) ;


NDorbit* NDorbitCombinatoricSum ( NDorbit *out , NDorbit *in ) ;
NDtmesh * NDtmeshCombinatoricSum ( NDorbit *circle , NDorbit *centers ) ;
NDtmesh * NDctmeshCombinatoricSum ( NDorbit *circle , NDorbit *centers ) ;
NDstmesh * NDscombinatoric_sum ( NDorbit *circle , NDorbit *centers ) ;
NDstmesh* NDclosed_scombinatoric_sum ( NDorbit *circle , NDorbit *centers ) ;

/******************************************************************************
 ******************************************************************************
 ************** Contents of numde_dynamics.c **********************************
 ******************************************************************************
 *****************************************************************************/
/* Now a giant step for men, a small step for the mankind */
void ND_RungeKutta ( NDorbit *out , unsigned long position , double tj , double *kk , vfield function) ;
NDorbit* run_RungeKutta ( unsigned long int nsteps , unsigned dimension , double *in , double tj , vfield function) ;

/* The prototypes of vfields used in simulations. */
void Lorenz ( double *out , unsigned dimension , const double *in , double a , double b , double c ) ;
void CLorenz ( double *out , unsigned dimension , const double *in ) ;
void Rossler ( double *out , unsigned dimension , const double *in ) ;  


/* Unstable part */
NDstmesh * NDTestSum ( NDorbit *centers , NDorbit *radius , void (*vectorf)(double * , double * )) ;
NDstmesh* NDTestSumP ( NDorbit *centers , NDorbit *radius, double *parameter , void (*vectorf)(double * , double )) ;


/* vector calculus */
double Dot ( double *va , double *vb , unsigned dimension ) ;
void Normalization( double *vector , unsigned dimension ) ;
double* Projection( double *vector , unsigned dimension , unsigned ortdir ) ;

void EulerAngles ( const NDorbit *radius , double *vector , NDorbit *out ) ;

void unstableEulerAngles (const NDorbit *radius , double *oldvector , double *newvector , NDorbit *out ) ;
double * unstableProjection ( double *vector ,       /* the vector to project */ 
			      unsigned dimension ,   /* the dimension of the overall space */
			      unsigned pdimension ,  /* the dimension of the projection space */ 
			      const double *basis ); /* the coordinates of the vectors in new basis */

/******************************************************************************
 ******************************************************************************
 ************** Contents of numde_matrix.c ************************************
 ******************************************************************************
 *****************************************************************************/

Matrix* mat_alloc(unsigned rows, unsigned cols) ;
void mat_free(Matrix *mat) ;

Matrix* mat_read_from_file(const char *filename) ;
int mat_write_into_file(const char *filename, const Matrix *mat) ;

Matrix* mat_multiply(Matrix *a, Matrix *b) ;

Matrix* mat_rotation ( unsigned dimension , double angle , unsigned coord_a , unsigned coord_b ) ;
void rotation_into_NDorbit ( NDorbit *orb , double angle , unsigned coord_a , unsigned coor_b ) ;
void mat_NDorbit ( NDorbit *orb , Matrix *matrix ) ;



#endif
