/*
 * AUTHOR: Gonçalo Morais (gnrm@fct.unl.pt) 
 *
 * Version :: 0.0.07 
 * STATE :: Highly Unstable.
 *
 * NUMDE is a scientific
 * library and pretends to implement the state-of-art of the
 * algorithms related with Differential Equations.  Although this
 * project is mainly written by one person, it received the valuable
 * contribution of several people. See the home page of the project
 * for further details.
 */

/*
 *  This file is part of NUMDE.
 *
 *  NUMDE is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  NUMDE is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with NUMDE.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef _NUMDE_POVRAY_C__
#define _NUMDE_POVRAY_C__

#include "numde.h"
#include "numde_matrix.h"
#include "numde_povray.h"

/********************************************************************
 ********************************************************************
 ********************************************************************
 * The next section is the definition of functions used to produce an
 * output for POVRAY
 ********************************************************************
 ********************************************************************
 ********************************************************************/

void povray_Include (FILE *out,char *pack){
  fprintf(out,"#include \"%s\"\n",pack);
}

void povray_add_Camera(FILE *out, TDvector pos, TDvector look, TDvector rot)
{
  assert(out!=NULL);

  fprintf(out,
	  "camera {\n"
	  "location < %f , %f , %f >\n"
	  "look_at < %f , %f , %f >\n"
	  "rotate < %f , %f , %f >\n"
	  "}\n\n",
	  pos.x,pos.y,pos.z,
	  look.x,look.y,look.z,
	  rot.x,rot.y,rot.z);
}


void povray_add_AreaLS(FILE *out, TDvector dir, Color color)
{
  assert(out!=NULL);

  fprintf(out,
	  "light_source {\n"
	  "<%f, %f, %f >*4\n"
	  "color rgb <%.2f, %.2f, %.2f>\n"
	  "area_light\n"
	  "2*x, 2*z,\n"
	  "4, 4\n"
	  "circular\n"
	  "orient\n"
	  "adaptive 1\n"
	  "jitter}\n\n",
	  dir.x,dir.y,dir.z,color.r,color.g,color.b);
}

void povray_NDorbit_sphere(FILE *out, NDorbit *in, double radius , char *texture )
{
  assert(out != NULL && in != NULL);

  unsigned dimension = in->dim ;

  unsigned long int i;
  for(i = 0 ; i < in->number ; i++)
    fprintf(out,
	     "sphere {\n"
	     "<%f, %f, %f >, %f\n"
	     "texture { %s }\n"
	     "finish {\n"
	     "phong .5 }\n"
	    "}\n",	    
	    in->mem[i*dimension] ,
	    in->mem[i*dimension + 1],
	    in->mem[i*dimension + 2],
	    radius , 
	    texture );
}

/* Just add a beautiful sky to our scene. */
void povray_Sky ( FILE *out , double *scale )
{
    fprintf(out,
	  "sky_sphere {\n"
	  "pigment {\n"
	  "gradient <1,1,1>\n"
	  "color_map {\n"
	  "[ 0.0 color rgb <0.99, 0.99, 1> ]\n"
	  "[ 1.0 color rgb <0.1, 0.1, 0.7> ]\n"
	  "}\n"
	  "scale <%f, %f, %f>\n"
	  "}\n"
	    "}\n",
	    scale[0] , scale[1] , scale[2] );
}



tubular_NDmesh * tubular_NDmesh_alloc ( unsigned long ncirc , 
					unsigned nrad , 
					unsigned dimension )
{ 
  /* 
   * To produce a tubular mesh, we need at least 3 points in each
   * circle and at least 2 circles.
   */
  if ( nrad < 3 || ncirc < 2 ){
    printf("ERROR: It is not possible to produce a tubular mesh with these values.\n") ;
    return ( NULL ) ;
  }
  
  unsigned triside = 3 ; /* the number of sides in a triangle */

  tubular_NDmesh *out ;
  out = malloc ( sizeof ( tubular_NDmesh )) ;
  
  out->dim = dimension ;
  out->rad = nrad ;
  out->circ = ncirc ;
  out->nvert = nrad * ncirc ;
  out->ntri = 2 * nrad * ( ncirc - 1 ) ;

  out->coord = calloc ( dimension * out->nvert , sizeof(double) );
  out->tri   = calloc ( triside * out->ntri , sizeof(unsigned long)) ; 
  
  if ( out == NULL ) 
    return NULL ;

  return out ;
}

tubular_NDmesh* tubular_closed_NDmesh_alloc ( unsigned long ncirc , 
					       unsigned nrad , 
					       unsigned dimension )
{ 
  /* 
   * To produce a tubular mesh, we need at least 3 points in each
   * circle and at least 2 circles.
   */
  if ( nrad < 3 || ncirc < 2 ){
    printf("ERROR: It is not possible to produce a tubular mesh with these values.\n") ;
    return ( NULL ) ;
  }
  
  unsigned triside = 3 ; /* the number of sides in a triangle */

  tubular_NDmesh *out ;
  out = malloc ( sizeof ( tubular_NDmesh )) ;
  
  out->dim = dimension ;
  out->rad = nrad ;
  out->circ = ncirc ;
  out->nvert = nrad * ncirc ;
  out->ntri = 2 * nrad * ncirc ;

  out->coord = calloc ( dimension * out->nvert , sizeof(double) );
  out->tri   = calloc ( triside * out->ntri , sizeof(unsigned long)) ; 
  
  if ( out == NULL ) 
    return NULL ;

  return out ;
}

stubular_NDmesh * stubular_closed_NDmesh_alloc ( unsigned long ncirc , 
						 unsigned nrad , 
						 unsigned dimension )
{ 
  /* 
   * To produce a tubular mesh, we need at least 3 points in each
   * circle and at least 2 circles.
   */
  if ( nrad < 3 || ncirc < 2 ){
    printf("ERROR: It is not possible to produce a tubular mesh with these values.\n") ;
    return ( NULL ) ;
  }
  
  unsigned triside = 3 ; /* the number of sides in a triangle */

  stubular_NDmesh *out ;
  out = malloc ( sizeof ( stubular_NDmesh )) ;
  
  out->dim = dimension ;
  out->rad = nrad ;
  out->circ = ncirc ;
  out->nvert = nrad * ncirc ;
  out->ntri = 2 * nrad * ncirc ;
  out->nnvec = out->nvert ;
    
  out->coord = calloc ( dimension * out->nvert , sizeof(double) );
  out->tri   = calloc ( triside * out->ntri , sizeof(unsigned long)) ;
  out->nvec  = calloc ( dimension * out->nnvec , sizeof( double ) ) ;
  
  if ( out == NULL ) 
    return NULL ;

  return out ;
}


stubular_NDmesh * stubular_NDmesh_alloc ( unsigned long ncirc , 
					  unsigned nrad , 
					  unsigned dimension )
{ 
  /* 
   * To produce a tubular mesh, we need at least 3 points in each
   * circle and at least 2 circles.
   */
  if ( nrad < 3 || ncirc < 2 ){
    printf("ERROR: It is not possible to produce a tubular mesh with these values.\n") ;
    return ( NULL ) ;
  }
  
  unsigned triside = 3 ; /* the number of sides in a triangle */

  stubular_NDmesh *out ;
  out = malloc ( sizeof ( stubular_NDmesh )) ;
  
  out->dim = dimension ;
  out->rad = nrad ;
  out->circ = ncirc ;
  out->nvert = nrad * ncirc ;
  out->ntri = 2 * nrad * ( ncirc - 1 ) ;
  out->nnvec = out->nvert ;
    
  out->coord = calloc ( dimension * out->nvert , sizeof(double) );
  out->tri   = calloc ( triside * out->ntri , sizeof(unsigned long)) ;
  out->nvec  = calloc ( dimension * out->nnvec , sizeof( double ) ) ;
  
  if ( out == NULL ) 
    return NULL ;

  return out ;
}


/* serves perfectly for open and closed mesh's */
void tubular_NDmesh_free ( tubular_NDmesh *in )
{
  if ( in == NULL )
    return ;

  free ( in->coord ) ;
  free ( in->tri ) ;
  free ( in ) ;
}

void stubular_NDmesh_free ( stubular_NDmesh *in )
{
  if ( in == NULL )
    return ;

  free ( in->coord ) ;
  free ( in->tri ) ;
  free ( in->nvec ) ;
  free ( in ) ;
}


void tubular_NDmesh_coord_print ( tubular_NDmesh *mesh )
{
  assert ( mesh != NULL ) ;

  unsigned long int i ;
  unsigned int j ;

  /* print the coordinate of the points */ 
  for ( i = 0 ; i < mesh->nvert ; i++ ){
    printf("\n") ;  
    for ( j = 0 ; j < mesh->dim ; j++ ){ 
	printf("%f " , mesh->coord[i * mesh->dim + j]  ) ;
      }
    }
  
  /* put another line at the end */
  printf("\n") ;
}

void tubular_NDmesh_tri_print ( tubular_NDmesh *mesh )
{
  assert ( mesh != NULL ) ;

  unsigned int columns = 3 ;
  
  unsigned long int i ;
  unsigned int j ;

  /* print the coordinate of the points */ 
  for ( i = 0 ; i < mesh->ntri ; i++ ){
    printf("\n") ;  
    for ( j = 0 ; j < columns ; j++ ){ 
	printf("%ld " , mesh->tri[i*columns + j]  ) ;
      }
    }
  
  /* put another line at the end */
  printf("\n") ;
}


tubular_NDmesh * NDcombinatoric_sum ( NDorbit *circle , NDorbit *centers )
{
  if ( circle->dim != centers->dim ){
    printf("ERROR: You are trying to create a mesh from structures with different dimensions.\n" );
    return NULL ;
  }

  unsigned triside = 3 ; /* the number of sides in a triangle! */

  tubular_NDmesh *out = tubular_NDmesh_alloc( centers->number , circle->number , circle->dim );  

  unsigned long i , j ;
  unsigned k ;
  
/*   The next instruction, that is the body of this function, follows 
 *   exactly the same spirit of the similar instruction in function 
 *   NDorbit_combinatoric_sum. Please read it carefully. 
 *
 *   In the next few lines we fill up the coordinates of the mesh.
 */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < circle->number ; i++ )
      for ( k = 0 ; k < circle->dim ; k++ )
	out->coord[( i + j * circle->number ) * circle->dim + k ] =
	  circle->mem[i * circle->dim + k] + centers->mem[j * centers->dim + k] ;

  
  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  return out ;
}


tubular_NDmesh * NDclosed_combinatoric_sum ( NDorbit *circle , NDorbit *centers )
{
  if ( circle->dim != centers->dim ){
    printf("ERROR: You are trying to create a mesh from structures with different dimensions.\n" );
    return NULL ;
  }

  unsigned triside = 3 ; /* the number of sides in a triangle! */

  tubular_NDmesh *out = tubular_closed_NDmesh_alloc( centers->number , circle->number , circle->dim );  

  unsigned long i , j ;
  unsigned k ;
  
/*   The next instruction, that is the body of this function, follows 
 *   exactly the same spirit of the similar instruction in function 
 *   NDorbit_combinatoric_sum. Please read it carefully. 
 *
 *   In the next few lines we fill up the coordinates of the mesh.
 */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < circle->number ; i++ )
      for ( k = 0 ; k < circle->dim ; k++ )
	out->coord[( i + j * circle->number ) * circle->dim + k ] =
	  circle->mem[i * circle->dim + k] + centers->mem[j * centers->dim + k] ;

  
  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  /* now we will add more triangles to close the mesh */
  j = out->circ - 1 ;
  
  for ( k = 0 ; k < out->rad ; k++ )
    {
      if ( k < out->rad - 1 ) 
	{
	  out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 2] = k  ;
	  
	  out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 ;
	  out->tri[2 * triside * ( k + j * out->rad) + 5] = k ;
	}
    }

  return out ;
}


stubular_NDmesh * NDscombinatoric_sum ( NDorbit *circle , NDorbit *centers )
{
  if ( circle->dim != centers->dim ){
    printf("ERROR: You are trying to create a mesh from structures with different dimensions.\n" );
    return NULL ;
  }

  unsigned triside = 3 ; /* the number of sides in a triangle! */

  stubular_NDmesh *out = stubular_NDmesh_alloc( centers->number , circle->number , circle->dim );  

  unsigned long i , j ;
  unsigned k ;
  
/*   The next instruction, that is the body of this function, follows 
 *   exactly the same spirit of the similar instruction in function 
 *   NDorbit_combinatoric_sum. Please read it carefully. 
 *
 *   In the next few lines we fill up the coordinates of the mesh.
 */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < circle->number ; i++ )
      for ( k = 0 ; k < circle->dim ; k++ )
	out->coord[( i + j * circle->number ) * circle->dim + k ] =
	  circle->mem[i * circle->dim + k] + centers->mem[j * centers->dim + k] ;

  
  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  /* now the normal vectors */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < circle->number ; i++ )
      for ( k = 0 ; k < circle->dim ; k++ )
	out->nvec[( i + j * circle->number ) * circle->dim + k ] =
	  out->coord[( i + j * circle->number ) * circle->dim + k ] - centers->mem[j * centers->dim + k] ;

  return out ;
}



void povray_mesh2 ( FILE *outfile , tubular_NDmesh *mesh )
{

  if ( outfile == NULL || mesh == NULL )
    return ;

  unsigned long i ;
  unsigned triside = 3 ;

  fprintf( outfile , 
	   "mesh2 {\n"
	   "   vertex_vectors {\n"
	   "   %ld,\n",
	   mesh->nvert ) ;
  
  
  for ( i = 0 ; i < mesh->nvert ; i++ ){
    if ( i < mesh->nvert - 1 )
      fprintf( outfile ,
	       "    < %f , %f , %f >,\n",
	       mesh->coord[i * triside ] ,   
	       mesh->coord[i * triside + 1] ,   
	       mesh->coord[i * triside + 2] ) ;
    else 
      fprintf( outfile ,
	       "    < %f , %f , %f >\n",
	       mesh->coord[i * triside ] ,   
	       mesh->coord[i * triside + 1] ,   
	       mesh->coord[i * triside + 2] ) ;
  }

  fprintf( outfile , 
	   " }\n  face_indices {\n"
	   "   %ld,\n",
	   mesh->ntri ) ;

  for ( i = 0 ; i < mesh->ntri ; i++ ){
    if ( i < mesh->ntri - 1 )
      fprintf( outfile ,
	       "    < %ld , %ld , %ld >,\n",
	       mesh->tri[i * triside ] ,   
	       mesh->tri[i * triside + 1] ,   
	       mesh->tri[i * triside + 2] ) ;
    else 
      fprintf( outfile ,
	       "    < %ld , %ld , %ld >\n",
	       mesh->tri[i * triside ] ,   
	       mesh->tri[i * triside + 1] ,   
	       mesh->tri[i * triside + 2] ) ;
  }

  fprintf( outfile ,
	   " }\n"
	   "pigment { color rgb<.7,.3,.9>}\n"
	   "finish { phong .5 phong_size 70 \n"
	   "        diffuse .7\n"
	   "        metallic }\n"
	   "\n}" );

}

/* other textures */
/* {T_Wood1} */
/* texture {T_Wood1 scale 2} */
/* pigment { color rgb<.7,.3,.9>}*/

/* now the function to write down a smooth mesh */
void povray_smesh2 ( FILE *outfile , stubular_NDmesh *mesh )
{

  if ( outfile == NULL || mesh == NULL )
    return ;

  unsigned long i ;
  unsigned triside = 3 ;
  
  /* first the coordinates of the vertices */ 
  fprintf( outfile , 
	   "mesh2 {\n"
	   "   vertex_vectors {\n"
	   "   %ld,\n",
	   mesh->nvert ) ;
  
  
  for ( i = 0 ; i < mesh->nvert ; i++ ){
    if ( i < mesh->nvert - 1 )
      fprintf( outfile ,
	       "    < %f , %f , %f >,\n",
	       mesh->coord[i * triside ] ,   
	       mesh->coord[i * triside + 1] ,   
	       mesh->coord[i * triside + 2] ) ;
    else 
      fprintf( outfile ,
	       "    < %f , %f , %f >\n",
	       mesh->coord[i * triside ] ,   
	       mesh->coord[i * triside + 1] ,   
	       mesh->coord[i * triside + 2] ) ;
  }
  
  /* second the coordinates of the normal vectors */
  fprintf( outfile , 
	   " }\n  normal_vectors {\n"
	   "   %ld,\n",
	   mesh->nnvec ) ;
  
  for ( i = 0 ; i < mesh->nnvec ; i++ ){
    if ( i < mesh->nnvec - 1 )
      fprintf( outfile ,
	       "    < %f , %f , %f >,\n",
	       mesh->nvec[i * triside ] ,   
	       mesh->nvec[i * triside + 1] ,   
	       mesh->nvec[i * triside + 2] ) ;
    else 
      fprintf( outfile ,
	       "    < %f , %f , %f >\n",
	       mesh->nvec[i * triside ] ,   
	       mesh->nvec[i * triside + 1] ,   
	       mesh->nvec[i * triside + 2] ) ;
  }
 
  
  /* finnaly the triangles */
  fprintf( outfile , 
	   " }\n  face_indices {\n"
	   "   %ld,\n",
	   mesh->ntri ) ;
  
  for ( i = 0 ; i < mesh->ntri ; i++ ){
    if ( i < mesh->ntri - 1 )
      fprintf( outfile ,
	       "    < %ld , %ld , %ld >,\n",
	       mesh->tri[i * triside ] ,   
	       mesh->tri[i * triside + 1] ,   
	       mesh->tri[i * triside + 2] ) ;
    else 
      fprintf( outfile ,
	       "    < %ld , %ld , %ld >\n",
	       mesh->tri[i * triside ] ,   
	       mesh->tri[i * triside + 1] ,   
	       mesh->tri[i * triside + 2] ) ;
  }
  
  fprintf( outfile ,
	   " }\n"
	   "texture { Aluminum }\n"
	   "finish { phong .5 phong_size 70 \n"
	   "        diffuse .7\n"
	   "        metallic }\n"
	   "\n}" );
  
}

/* "pigment { color rgb<.7,.3,.9>}\n" */

#endif 
