/*
 * AUTHOR: Gonçalo Morais (gnrm@fct.unl.pt) 
 *
 * Version :: 0.0.07 
 * STATE :: Highly Unstable.
 *
 * NUMDE is a scientific
 * library and pretends to implement the state-of-art of the
 * algorithms related with Differential Equations.  Although this
 * project is mainly written by one person, it received the valuable
 * contribution of several people. See the home page of the project
 * for further details.
 */

/*
 *  This file is part of NUMDE.
 *
 *  NUMDE is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  NUMDE is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with NUMDE.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef _NUMDE_MATRIX_C__
#define _NUMDE_MATRIX_C__

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "numde.h"
#include "numde_matrix.h"

/* A macro to find the entry ij in the matrix m */
#define mat(m,i,j)  m->mem[i * m->cols + j]
/* The address of the entry ij in the matrix m*/
#define matp(m,i,j) (m->mem + i * m->cols + j) 

Matrix* mat_alloc(unsigned rows, unsigned cols)
{
  Matrix *mat;

  mat = malloc(sizeof(Matrix));
  if (mat == NULL) {
    return NULL;
  }
  mat->rows = rows;
  mat->cols = cols;
  mat->mem = calloc(rows * cols, sizeof(double));

  if (mat->mem == NULL) {
    free(mat);
    return NULL;
  }

  return mat;
}



void mat_free(Matrix *mat)
{
  if (mat == NULL)
    return;
  free(mat->mem);
  free(mat);
  return;
}



Matrix* mat_read_from_file(const char *filename)
{
  FILE *fd;
  char line[512];
  unsigned i, j, rows, cols;
  Matrix *mat;

  fd = fopen(filename, "r");
  if (fd == NULL) {
    return NULL;
  }

  line[0] = 0;
  fgets(line, 512, fd);
  if (sscanf(line, "%u %u", &rows, &cols) != 2) {
    printf("error: There is no information about the number of rows or columns.\n");
  }

  mat = mat_alloc(rows, cols);
  for (i=0 ; i<rows ; ++i) {
    for (j=0 ; j<cols ; ++j) {
      if (fscanf(fd, "%lf", matp(mat, i, j)) != 1) {
	printf("error: The matrix is not complete.\n");
	mat_free(mat);
	return NULL;
      }
    }
  }
  fclose(fd);
    
  return mat;
}



int mat_write_into_file(const char *filename, const Matrix *mat)
{
  FILE *fd;
  unsigned i, j;

  if (mat == NULL)
    return 0;
  
  fd = (filename == NULL) ? stdout : fopen(filename, "w");
  if (fd == NULL) {
    return 0;
  }

  printf("%u %u\n", mat->rows, mat->cols);
  for (i=0 ; i<mat->rows ; ++i) {
    for (j=0 ; j<mat->cols ; ++j) 
      printf("%lf ", mat(mat, i, j));
    printf("\n");
  }

  if (filename != NULL)
    fclose(fd);
  return 1;
} 



Matrix* mat_multiply(Matrix *a, Matrix *b)
{
  unsigned i, j, k;
  Matrix *m;

  if (a == NULL || b ==NULL)
    return NULL;


  if (a->cols != b->rows){
    printf( "\nThe product of these 2 matrices is not possible.\n" );
    printf( "The product A*B is only possible if columns(A)=rows(B).\n\n") ;
    return NULL;
  }


  m = mat_alloc(a->rows, b->cols);
  for( i = 0 ; i < a->rows ; i++ ){
    for( j = 0 ; j < b->cols ; j++ ){
      for( k = 0 ; k < a->cols ; k++ ){
	mat(m,i,j) +=  mat(a,i,k) * mat(b,k,j);
      }
    }
  }

  return m;
}

/*******************************************************************************
 *******************************************************************************
 * Now that we have the basic structure to work with matrices, we may
 * develop the thing a litle more, introducing all the wonderful
 * properties that we are used to see in Linear Algebra.
 *******************************************************************************
 ******************************************************************************/

/* The next function produces a pointer to a matrix, that may be used
   to compute the n-dimensional rotation of a angle theta, in the
   plane generated i<j components of eucledian space. */
Matrix* mat_rotation ( unsigned dimension , 
		       double angle ,
		       unsigned coord_a ,
		       unsigned coord_b )
{
  if ( coord_a == coord_b )
    printf("A rotation must be perform in a plane.\n"
	   "You only gave one component.\n");
  
  Matrix *rotation = mat_alloc( dimension , dimension ) ;
  
  unsigned i ;
  for ( i = 0 ; i < dimension ; i++ )
    mat(rotation,i,i) = 1 ;
  
  mat(rotation , coord_a , coord_a) = cos(angle) ;
  mat(rotation , coord_a , coord_b) = -sin(angle) ;
  mat(rotation , coord_b , coord_a) = sin(angle) ;
  mat(rotation , coord_b , coord_b) = cos(angle) ;
  
  return rotation ;
}  

/* 
 * The problems that we want to solve have always a certain
 * dimensional structure. Indeed the array of doubles in the mem
 * component of a NDorbit, are always seen as an array of vectors. The
 * dimension of these vectors are only related with the space where
 * the vector field is defined. This said, is quite natural that we
 * define linear transformations over this structure. The next
 * function rotates all the elements of a NDorbit.

 * This operation is supported in a n-dimensional space, so the
 * arguments are not the components that are invariant, but precisely
 * those where the transformation is operated. */

void rotation_into_NDorbit ( NDorbit *orb , double angle , unsigned coord_a , unsigned coord_b )
{
  Matrix *rotation = mat_rotation( orb->dim , angle , coord_a , coord_b ) ;
  Matrix *help = mat_alloc ( orb->dim , 1 ) ;
  
  unsigned i ;
  for ( i = 0 ; i < orb->dim ; i++ )
    help->mem[i] = orb->mem[i] ;

  unsigned long j ;
  
  for ( j = 1 ; j < orb->number ; j++ ){
    help = mat_multiply(rotation , help ) ;
    for ( i = 0 ; i < orb->dim ; i++ )
      orb->mem[j * orb->dim + i] = help->mem[i] ;
  }

  mat_free(help) ;
  mat_free(rotation) ;
}    

/*
 * The next function is the natural generalization of the rotation
 * version to any matrix. Notice also that the operation is made using
 * homogeneus coordinates for higher flexibility. Such a matrix will
 * be of the type (in $\mathbb{R}^3$)
 * 
 * r11 r12 r13 tx
 * r21 r22 r23 ty
 * r31 r32 r33 tz
 *   0   0   0  1
 * 
 * A vector in this coordinates is of the type (x,y,z,1).
 */ 

void mat_NDorbit ( NDorbit *orb , Matrix *matrix )
{
  Matrix *temp = mat_alloc( orb->dim + 1 , 1 ) ;
  
  unsigned i ;
  for ( i = 0 ; i < orb->dim ; i++ )
    temp->mem[i] = orb->mem[i] ;
  orb->mem[orb->dim] = 1 ; /* the last entry of homogeneous coordinates is non-zero. */
  
  unsigned long j ;
  for ( j = 1 ; j < orb->number ; j++ ){
    temp = mat_multiply( matrix , temp ) ;
    for ( i = 0 ; i < orb->dim ; i++ )
      orb->mem[j * orb->dim + i] = temp->mem[i] ;
  }
   
  mat_free(temp) ;
}


#endif
