/*
 * AUTHOR: Gonçalo Morais (gnrm@fct.unl.pt) 
 *
 * Version :: 0.0.07 
 * STATE :: Highly Unstable.
 *
 * NUMDE is a scientific
 * library and pretends to implement the state-of-art of the
 * algorithms related with Differential Equations.  Although this
 * project is mainly written by one person, it received the valuable
 * contribution of several people. See the home page of the project
 * for further details.
 */

/*
 *  This file is part of NUMDE.
 *
 *  NUMDE is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  NUMDE is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with NUMDE.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef __NUMDE_C__
#define __NUMDE_C__

#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <string.h>

#include "numde.h"
#include "numde_matrix.h"

/*****************************************************************************
 *****************************************************************************
 * The first functions are necessery to alloc and free memory for
 * structure NDorbit.
 *****************************************************************************
 ****************************************************************************/

NDorbit* NDorbit_alloc ( unsigned long int nelements , unsigned int dimension )
{
  NDorbit *out ;

  out = malloc ( sizeof( NDorbit ) ) ;

  if ( out == NULL )
    return NULL ;

  out->dim = dimension ;
  out->number = nelements ;
  out->mem =  calloc( dimension * nelements , sizeof( double ) ) ;

  if ( out -> mem == NULL ) 
    return NULL ;

  return out ;
}

void NDorbit_free ( NDorbit *in ){
  if ( in == NULL ) 
    return ;
  free(in -> mem) ;
  free(in) ;
}

/* An NDorbit is just an array of doubles. But this array has more
   structure, because is indeed an array of vectors with dimension
   dim. The next macro help us to point the beginning of each of these
   vectors. */
#define NDp(in,i) (in->mem+(i)*in->dim)    


/*****************************************************************************
 *****************************************************************************
 * We need to produce a structured output (print) for the structures
 * defined.
 *****************************************************************************
 ****************************************************************************/

void NDorbit_print ( NDorbit *in )
{
  double *nd ;
  
  unsigned long int rows = in -> number ;
  unsigned int columns = in -> dim ;

  unsigned long int i ;
  unsigned int j ;

  assert ( in != NULL ) ;
  for ( i = 0 ; i < rows ; i++ ){
    nd=NDp(in,i) ;
    printf("\n") ;  
    for ( j = 0 ; j < columns ; j++ ){ 
	printf("%f " , nd[j] ) ;
      }
    }
  
  /* put another line at the end */
  printf("\n") ;
}

void NDorbit_fprint ( char *filename , NDorbit *in )
{
  /* Warning: We assume that the file is not already open */
  FILE *outfile ;
  outfile = fopen( filename , "w" ) ;


  double *nd ;
  
  unsigned long int rows = in -> number ;
  unsigned int columns = in -> dim ;

  unsigned long int i ;
  unsigned int j ;

  assert ( in != NULL ) ;
  for ( i = 0 ; i < rows ; i++ ){
    nd=NDp(in,i) ;
    fprintf(outfile , "\n") ;  
    for ( j = 0 ; j < columns ; j++ ){ 
      fprintf(outfile, "%f " , nd[j] ) ;
    }
  }
  
  fclose(outfile) ;
}


/*****************************************************************************
 *****************************************************************************
 * We can induce a vector structure in a NDorbit, by create a NDorbit
 * of the difference between two consecutive points in the original
 * NDorbit. 
 *****************************************************************************
 ****************************************************************************/

NDorbit* NDorbit_diff ( NDorbit *in )
{
  NDorbit *out =NDorbit_alloc ( in->number - 1 , in->dim ) ; 
  
  unsigned long i , nvectors = in->number / in->dim ;
  unsigned j ;

  for ( i = 0 ; i < nvectors ; i++ )
    for ( j = 0 ; j < in->dim ; j++ )
      out->mem[i*in->dim + j] = in->mem[(i + 1)*in->dim + j] - in->mem[i*in->dim + j] ;

  return out ;
}

/* 
 * NDorbit_combinatoric_sum
 * We do control the dimention of both orbits to see if they
 * match. Carefull must be taken in order to do not sum things that
 * may be quite absurd. We emit an error message and return NULL.
 *
 * This function just sum every "vector" in NDorbit *trans to every
 * "point" in NDorbit *ref.
 *
 */

NDorbit* NDorbit_combinatoric_sum ( NDorbit *ref , NDorbit *trans )
{

  if ( ref->dim != trans->dim ){
    printf("ERROR: You are trying to sum NDorbit's with different dimension.\n"
	   "The computation will not continue.\n" 
	   "Please read the documentation of NDorbit_sum in NUMDE library.\n");
    return NULL ;
  }
  
  NDorbit *out = NDorbit_alloc( ref->number * trans->number , ref->dim ) ; 
  
  unsigned long i , j ;
  unsigned k ;
  
  /*
   * There are several precautions that we had while writing the
   * following code.
   *
   * We had two options. The first was to compute the translations of
   * each vector of *ref by all vectors in *trans. The second was to
   * compute the translation of *ref by a single vector in *trans one
   * at the time. We prefered the second option. The reason is quite
   * obvious, thinking that we want to use this reasoning building a
   * mesh of the tube. Put all the vectors in the right order is very
   * important.
   * 
   * This way, imagine that *ref has 5 vectors, r0, ... ,r4 , and
   * *trans has 3 vectors, t0,t1,t2. Then *out will have 3x5 vectors,
   * that can be placed in a matrix by
   *
   * out = r0+t0 r1+t0 r2+t0 r3+t0 r4+t0
   *       r0+t1 r1+t1 r2+t1 r3+t1 r4+t1
   *       r0+t2 r1+t2 r2+t2 r3+t2 r4+t2
   *
   * Searching by rows in this matrix, the k-component of vector
   * out[i,j], that is the sum of vector ri with vector tj, will be
   * placed in out[(i+jx5)*dim + k]. This is the spirit of the
   * following (obscure) instruction. Classical, but always beautiful.
   *
   */

  for (j = 0 ; j < trans->number ; j++ )
    for ( i = 0 ; i < ref->number ; i++ )
      for ( k = 0 ; k < ref->dim ; k++ )
	out->mem[(i+ j * ref->number) * ref->dim + k] = ref->mem[i * ref->dim + k] + trans->mem[ j * trans->dim + k] ; 
  
  return out ;
}


/* 
 *  The function NDcombinatoric_sum works perfectly well in situations
 *  where we want to create a tubular mesh for things with curvuture
 *  equal to 0. In all other cases we need more careful and we must add
 *  an information about how do things are curving.
 */

stubular_NDmesh * NDclosed_scombinatoric_sum ( NDorbit *circle , NDorbit *centers )
{
  if ( circle->dim != centers->dim ){
    printf("ERROR: You are trying to create a mesh from structures with different dimensions.\n" );
    return NULL ;
  }

  unsigned triside = 3 ; /* the number of sides in a triangle! */

  stubular_NDmesh *out = stubular_closed_NDmesh_alloc( centers->number , circle->number , circle->dim );  

  unsigned long i , j ;
  unsigned k ;
  
/*   The next instruction, that is the body of this function, follows 
 *   exactly the same spirit of the similar instruction in function 
 *   NDorbit_combinatoric_sum. Please read it carefully. 
 *
 *   In the next few lines we fill up the coordinates of the mesh.
 */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < circle->number ; i++ )
      for ( k = 0 ; k < circle->dim ; k++ )
	out->coord[( i + j * circle->number ) * circle->dim + k ] =
	  circle->mem[i * circle->dim + k] + centers->mem[j * centers->dim + k] ;

  
  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  /* now we will add more triangles to close the mesh */
  j = out->circ - 1 ;
  
  for ( k = 0 ; k < out->rad ; k++ )
    {
      if ( k < out->rad - 1 ) 
	{
	  out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 2] = k  ;
	  
	  out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 ;
	  out->tri[2 * triside * ( k + j * out->rad) + 5] = k ;
	}
    }

  
  /* now the normal vectors */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < circle->number ; i++ )
      for ( k = 0 ; k < circle->dim ; k++ )
	out->nvec[( i + j * circle->number ) * circle->dim + k ] =
	  out->coord[( i + j * circle->number ) * circle->dim + k ] - centers->mem[j * centers->dim + k] ;

  return out ;
}

/* 
 * The following function (NDTestSum while it does not have a definite
   name) computes a tubular mesh around the points of the NDorbit
   *centers, perpendicular to the vector given at the same point, by
   the vector field */

stubular_NDmesh * NDTestSum ( NDorbit *centers , 
			      NDorbit *radius, 
			      void (*vectorf)(double * , double * ))
{
  unsigned triside = 3 ; /* the number of sides in a triangle! */
  /* the mesh that we want to produce */
  stubular_NDmesh *out = stubular_closed_NDmesh_alloc( centers->number , radius->number , radius->dim );
  
  /* just a structure to record the transformation by EulerAngles of
     original *radius */
  NDorbit *trad = NDorbit_alloc( radius->number , radius->dim ) ; 
  
  /* the vectors that will be used to compute the values of the vector field */
  double *tmpin  = calloc( centers->dim , sizeof(double) ) ;
  double *tmpout = calloc( centers->dim , sizeof(double) ) ;


  unsigned long i , j ;
  unsigned k ;
  
  /*   The next instruction, that is the body of this function, follows 
   *   exactly the same spirit of the similar instruction in function 
   *   NDorbit_combinatoric_sum. Please read it carefully. 
   *
   *   In the next few lines we fill up the coordinates of the mesh.
   */
  for ( j = 0 ; j < centers->number ; j++ ){
    for ( k = 0 ; k < centers->dim ; k++ ) 
      tmpin[k] = centers->mem[j * centers->dim + k] ;
    vectorf(tmpin,tmpout) ;
    EulerAngles( radius , tmpout , trad ) ;
    for ( i = 0 ; i < radius->number ; i++ )
      for ( k = 0 ; k < radius->dim ; k++ ) 
	out->coord[( i + j * radius->number ) * radius->dim + k ] =
	  trad->mem[i * radius->dim + k] + centers->mem[j * centers->dim + k] ;
  }

  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  /* now we will add more triangles to close the mesh */
  j = out->circ - 1 ;
  
  for ( k = 0 ; k < out->rad ; k++ )
    {
      if ( k < out->rad - 1 ) 
	{
	  out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 2] = k  ;
	  
	  out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 ;
	  out->tri[2 * triside * ( k + j * out->rad) + 5] = k ;
	}
    }

  
  /* now the normal vectors */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < radius->number ; i++ )
      for ( k = 0 ; k < radius->dim ; k++ )
	out->nvec[( i + j * radius->number ) * radius->dim + k ] =
	  out->coord[( i + j * radius->number ) * radius->dim + k ] - centers->mem[j * centers->dim + k] ;



  /* Like Tibet, we want free memory */
  NDorbit_free(trad) ;
  free(tmpin) ;
  free(tmpout) ;

  return out ;
}

/* This is exactly the same principle but generates a mesh from a
   vector field that is defined parametrically */
stubular_NDmesh* NDTestSumP ( NDorbit *centers , 
			      NDorbit *radius, 
			      double *parameter ,
			      void (*vectorf)(double * , double ))
{
  unsigned triside = 3 ; /* the number of sides in a triangle! */
  /* the mesh that we want to produce */
  stubular_NDmesh *out = stubular_closed_NDmesh_alloc( centers->number , radius->number , radius->dim );
  
  /* just a structure to record the transformation by EulerAngles of
     original *radius */
  NDorbit *trad = NDorbit_alloc( radius->number , radius->dim ) ; 
  
  /* the vectors that will be used to compute the values of the vector field */
  double *tmpout = calloc( centers->dim , sizeof(double) ) ;


  unsigned long i , j ;
  unsigned k ;
  
  /*   The next instruction, that is the body of this function, follows 
   *   exactly the same spirit of the similar instruction in function 
   *   NDorbit_combinatoric_sum. Please read it carefully. 
   *
   *   In the next few lines we fill up the coordinates of the mesh.
   */
  for ( j = 0 ; j < centers->number ; j++ ){
    vectorf(tmpout, parameter[j]) ;
    EulerAngles( radius , tmpout , trad ) ;
    for ( i = 0 ; i < radius->number ; i++ )
      for ( k = 0 ; k < radius->dim ; k++ ) 
	out->coord[( i + j * radius->number ) * radius->dim + k ] =
	  trad->mem[i * radius->dim + k] + centers->mem[j * centers->dim + k] ;
  }

  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  /* now we will add more triangles to close the mesh */
  j = out->circ - 1 ;
  
  for ( k = 0 ; k < out->rad ; k++ )
    {
      if ( k < out->rad - 1 ) 
	{
	  out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 2] = k  ;
	  
	  out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 ;
	  out->tri[2 * triside * ( k + j * out->rad) + 5] = k ;
	}
    }

  
  /* now the normal vectors */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < radius->number ; i++ )
      for ( k = 0 ; k < radius->dim ; k++ )
	out->nvec[( i + j * radius->number ) * radius->dim + k ] =
	  out->coord[( i + j * radius->number ) * radius->dim + k ] - centers->mem[j * centers->dim + k] ;



  /* Like Tibet, we want free memory */
  NDorbit_free(trad) ;
  free(tmpout) ;

  return out ;
}

void NDmeshCoordPrintST ( stubular_NDmesh *mesh )
{
  assert ( mesh != NULL ) ;

  unsigned long int i ;
  unsigned int j ;

  /* print the coordinate of the points */ 
  for ( i = 0 ; i < mesh->nvert ; i++ ){
    printf("\n") ;  
    for ( j = 0 ; j < mesh->dim ; j++ ){ 
	printf("%f " , mesh->coord[i * mesh->dim + j]  ) ;
      }
    }
  
  /* put another line at the end */
  printf("\n") ;
}

void EulerAngles (const NDorbit *radius , double *vector , NDorbit *out ) 
{
  double *xvector , *yvector , *zvector ;
  xvector = calloc( out->dim , sizeof(double)) ;
  yvector = calloc( out->dim , sizeof(double)) ;
  zvector = calloc( out->dim , sizeof(double)) ;
  xvector[0] = 1 ;
  yvector[1] = 1 ;  
  zvector[2] = 1 ;

  double *projection = Projection( vector , out->dim , 2 );
  Normalization( projection , out->dim ) ;
  Normalization( vector , out->dim ) ;

  /* the 2 Euler's angles necessary, the optimized idea thanks to Rui Rodrigues (rapr@fct.unl.pt) . */
  double theta = atan2( projection[1] , projection[0] );
  double phi = atan2( sqrt( pow(vector[0],2) + pow(vector[1],2)) , vector[2] ) ;

  Matrix *R = mat_multiply( mat_rotation ( out->dim , theta , 0 , 1 ) , mat_rotation ( out->dim , phi , 0 , 2 )) ;
  Matrix *tmp0 = mat_alloc ( out->dim , 1 ) ;
  Matrix *tmp1 = mat_alloc ( out->dim , 1 ) ;

  unsigned long i ;
  unsigned j ;

  for ( i = 0 ; i < out->number ; i++ ){
    for ( j = 0 ; j < out->dim ; j++ )
      tmp0->mem[j] = radius->mem[i * radius->dim + j] ;

    tmp1 = mat_multiply( R , tmp0 ) ;

    for ( j = 0 ; j < out->dim ; j++ )
      out->mem[i * out->dim + j] = tmp1->mem[j] ;
  }

  mat_free(R) ;
  mat_free(tmp0) ;
  mat_free(tmp1) ;
  free(xvector) ;
  free(yvector) ;
  free(zvector) ;

  free(projection) ;
}

double Dot ( double *va , double *vb , unsigned dimension ) 
{
  double out = 0 ;

  unsigned i ;
  for ( i = 0 ; i < dimension ; i++ )
    out += va[i] * vb[i] ;

  return out ;
}

void Normalization( double *vector , unsigned dimension )
{
  unsigned i ;
  double norm =  sqrt( Dot( vector , vector , dimension ) ) ;
  
  for ( i = 0 ; i < dimension ; i++ )
    vector[i] = vector[i] / ( norm + 0.0 ) ;

}


double * Projection( double *vector , unsigned dimension , unsigned ortdir )
{
  double *xvec = malloc( dimension * sizeof(double)) ;

  unsigned i ; 
  for ( i = 0 ; i < dimension ; i++ ){
    xvec[i] = 1 ;
  }
  
  for ( i = 0 ; i < dimension ; i++ )
    xvec[i] = vector[i] * xvec[i] ;

  xvec[ortdir] = 0 ;
 
  return xvec ;
}


/*****************************************************************************
 *****************************************************************************
 * When we are dealing with numerical algorithms to simulate a certain
 * behavior, it is good to produce a lot of points. However, we do not
 * need so many points when we want to represent the data, for
 * example, in a graphic. The next set of routines are used precisely
 * to discard some of the points.
 *****************************************************************************
 ****************************************************************************/

NDorbit* NDorbit_cut ( NDorbit *in , unsigned gap )
{
  if ( in == NULL )
    return NULL ;

  unsigned long newlength =  in->number / (gap + 0.0) - 1 ;
  unsigned dimension = in->dim ;
  NDorbit *out = NDorbit_alloc ( newlength , dimension ) ;

  unsigned long i ;
  unsigned j ;

  for ( i = 0 ; i < newlength ; i++ )
    for ( j = 0 ; j < dimension ; j++ )
      out->mem[i*dimension + j] = in->mem[i*dimension*gap + j] ;

  return out ;
  
}


/********************************************************************
 ********************************************************************
 ********************************************************************
 *
 * The next set of functions are the implementation of RUNGE-KUTTA of
 * order 4. The implementation of this algorithm was made independent
 * from the dimension of the problem.
 *
 ********************************************************************
 ********************************************************************
 *******************************************************************/

 
/* In the computation of the Runge-Kutta algorithm of fouth order, is
   necessary to use for vectors k1-k4, to save the values of the values
   of the vector field in 4 different points. As usual, this is simply an
   array of doubles, that is indeed an array of vectors of a certain
   dimension. The next macro allow us to refer to the beginning of each
   of these vectors. You should also note that the implementation
   reserves not 4 but 5 of these vectors. This is so simply because the
   first vector KK(0) is the place where the values are temporarily
   preserved (see macro RK_SUM bellow.)
*/ 
#define KK(i) (kk+i*dimension)  


/* In Runge Kutta algorithm of fourth order, in problems of dimension
   bigger than one, the algorithm computes the values of auxiliary
   vectors k1-k4 in a recurrent fashion. However we must preserve the
   original value of each of these vectors.

   To keep everything correct, we allocate one extra vector k0, and
   use it to carry on all the calculations. In this way, we preserve
   everything that we need. 

   The macro RK_SUM(i,dt) just defines a way to say that to the
   original vector "initial", we add dt*ki in the usual way.
*/
#define RK_SUM(i,dt)                  \
  do {                                \
  unsigned ii;                        \
  double *a, *b;                      \
  a = KK(0);                          \
  b = KK(i);                          \
  for (ii=0 ; ii<dimension ;++ii) {   \
    a[ii] = initial[ii] + dt * b[ii]; \
  }                                   \
  } while(0)




/* Now a giant step for men, a small step for the mankind */
void ND_RungeKutta ( NDorbit *out , 
		     unsigned long position , 
		     double tj ,
		     double *kk ,
		     vfield function) 
{
  unsigned int dimension = out->dim ;

  double *initial, *result , htj = tj/2. ;
  initial = NDp( out, position) ;
  result = NDp( out , position+1 ) ;

  /* first set of the algorithm */
  function( KK(1) , dimension , initial ) ;
  RK_SUM(1, htj);

  function( KK(2) , dimension , KK(0)) ;
  RK_SUM(2, htj);

  function( KK(3) , dimension , KK(0)) ;
  RK_SUM(3, tj);

  function( KK(4) , dimension , KK(0)) ;

  unsigned i ;
  for ( i = 0 ; i < dimension ; i++ )
    result[i] = initial[i] + tj * (kk[i+dimension] + 2 *kk[i+2*dimension] 
				   + 2* kk[i+3*dimension] + kk[i+4*dimension])/6. ;
}		

NDorbit* run_RungeKutta ( unsigned long int nsteps ,
			   unsigned dimension ,
			   double *in ,
			   double tj ,
			   vfield function)
{
  
  NDorbit *out = NDorbit_alloc ( nsteps , dimension ) ;
  double *kk;

  kk = malloc ( 5 * dimension * sizeof(double)) ;

  memcpy(NDp(out,0), in , dimension * sizeof(double)) ;

  unsigned long int i ;
  for ( i = 0 ; i < nsteps-1 ; i++ ){
    fprintf( stdout , "Evaluating step %lu of %lu\r" , i , nsteps ) ;
    ND_RungeKutta( out , i , tj , kk , function ) ;
  }

  free(kk) ;
  fprintf( stdout , "The orbit was computed by the Runge-Kutta algorithm of 4th order of NUMDE library.\n" ) ;
  fprintf( stdout , "NUMDE is free software (as in freedom). Please visit the homepage of the project for further information.\n" ) ;
  fprintf( stdout , "For bug report or suggestions, please contact Goncalo Morais (gnrm@fct.unl.pt).\n" ) ;
  return out ;
}

/********************************************************************
 ********************************************************************
 ********************************************************************
 * The next section is the definition of several classical vector
 * fields. Note that most part of them are implemented in two
 * stages. First we define the most general, with a whole range set of
 * parameters. After we define, if this makes sense, the classical
 * case.
 ********************************************************************
 ********************************************************************
 ********************************************************************/

/* the first is the lorenz vfield. this can not be used in runge-kutta
   directely because has not the structure of a vfield*/
void Lorenz ( double *out , unsigned dimension , const double *in , double a , double b , double c )
{
  out[0] = a * ( in[1] - in[0] ) ;
  out[1] = b * in[0] - in[1] - in[0] * in[2] ;
  out[2] = c * in[2] + in[0] * in[1] ;
}
/* Now the classical values for Lorenz Attractor (to be used in runge-kutta). */
void CLorenz ( double *out , unsigned dimension , const double *in )
{
  Lorenz( out , dimension , in , 10 , 28 , -8/3. ) ;
}

/* Equally famous is the case of the Rossler attractor. */
void Rossler ( double *out , unsigned dimension , const double *in )
{
  out[0] = -in[1] -in[2] ;
  out[1] = in[0] + .25 * in[1] ;
  out[2] = 1 + in[2] *( in[0] - 5.5 ) ;
}


  
#endif
