#include <stdio.h>
#include <stdlib.h>

#include "numde.h"  
#include "numde_matrix.h"

#define pi M_PI /* a more conventional name for \pi */ 

typedef struct {
  unsigned dim ;                /* the dimension of the space where mesh lives */
  unsigned rad ;                /* the number of points in each section */
  unsigned long circ ;          /* the number of sections (circles) */
  unsigned long nvert ;         /* the number of vertices */
  double *coord ;               /* where the coordinates of all this will be */
  unsigned long ntri ;          /* the number of triangles */
  unsigned long *tri ;          /* the indices of the vertex that belong to each triangle */
} tubular_NDmesh ;

tubular_NDmesh * tubular_NDmesh_alloc ( unsigned long ncirc , 
					unsigned nrad , 
					unsigned dimension )
{ 
  /* 
   * To produce a tubular mesh, we need at least 3 points in each
   * circle and at least 2 circles.
   */
  if ( nrad < 3 || ncirc < 2 ){
    printf("ERROR: It is not possible to produce a tubular mesh with these values.\n") ;
    return ( NULL ) ;
  }
  
  unsigned triside = 3 ; /* the number of sides in a triangle */

  tubular_NDmesh *out ;
  out = malloc ( sizeof ( tubular_NDmesh )) ;
  
  out->dim = dimension ;
  out->rad = nrad ;
  out->circ = ncirc ;
  out->nvert = nrad * ncirc ;
  out->ntri = 2 * nrad * ( ncirc - 1 ) ;

  out->coord = calloc ( dimension * out->nvert , sizeof(double) );
  out->tri   = calloc ( triside * out->ntri , sizeof(unsigned long)) ; 
  
  if ( out == NULL ) 
    return NULL ;

  return out ;
}

tubular_NDmesh * tubular_closed_NDmesh_alloc ( unsigned long ncirc , 
					       unsigned nrad , 
					       unsigned dimension )
{ 
  /* 
   * To produce a tubular mesh, we need at least 3 points in each
   * circle and at least 2 circles.
   */
  if ( nrad < 3 || ncirc < 2 ){
    printf("ERROR: It is not possible to produce a tubular mesh with these values.\n") ;
    return ( NULL ) ;
  }
  
  unsigned triside = 3 ; /* the number of sides in a triangle */

  tubular_NDmesh *out ;
  out = malloc ( sizeof ( tubular_NDmesh )) ;
  
  out->dim = dimension ;
  out->rad = nrad ;
  out->circ = ncirc ;
  out->nvert = nrad * ncirc ;
  out->ntri = 2 * nrad * ncirc ;

  out->coord = calloc ( dimension * out->nvert , sizeof(double) );
  out->tri   = calloc ( triside * out->ntri , sizeof(unsigned long)) ; 
  
  if ( out == NULL ) 
    return NULL ;

  return out ;
}


void tubular_NDmesh_free ( tubular_NDmesh *in )
{
  if ( in == NULL )
    return ;

  free ( in->coord ) ;
  free ( in->tri ) ;
  free ( in ) ;
}

void tubular_NDmesh_coord_print ( tubular_NDmesh *mesh )
{
  assert ( mesh != NULL ) ;

  unsigned long int i ;
  unsigned int j ;

  /* print the coordinate of the points */ 
  for ( i = 0 ; i < mesh->nvert ; i++ ){
    printf("\n") ;  
    for ( j = 0 ; j < mesh->dim ; j++ ){ 
	printf("%f " , mesh->coord[i * mesh->dim + j]  ) ;
      }
    }
  
  /* put another line at the end */
  printf("\n") ;
}

void tubular_NDmesh_tri_print ( tubular_NDmesh *mesh )
{
  assert ( mesh != NULL ) ;

  unsigned int columns = 3 ;
  
  unsigned long int i ;
  unsigned int j ;

  /* print the coordinate of the points */ 
  for ( i = 0 ; i < mesh->ntri ; i++ ){
    printf("\n") ;  
    for ( j = 0 ; j < columns ; j++ ){ 
	printf("%ld " , mesh->tri[i*columns + j]  ) ;
      }
    }
  
  /* put another line at the end */
  printf("\n") ;
}


tubular_NDmesh * NDcombinatoric_sum ( NDorbit *circle , NDorbit *centers )
{
  if ( circle->dim != centers->dim ){
    printf("ERROR: You are trying to create a mesh from structures with different dimensions.\n" );
    return NULL ;
  }

  unsigned triside = 3 ; /* the number of sides in a triangle! */

  tubular_NDmesh *out = tubular_NDmesh_alloc( centers->number , circle->number , circle->dim );  

  unsigned long i , j ;
  unsigned k ;
  
/*   The next instruction, that is the body of this function, follows 
 *   exactly the same spirit of the similar instruction in function 
 *   NDorbit_combinatoric_sum. Please read it carefully. 
 *
 *   In the next few lines we fill up the coordinates of the mesh.
 */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < circle->number ; i++ )
      for ( k = 0 ; k < circle->dim ; k++ )
	out->coord[( i + j * circle->number ) * circle->dim + k ] =
	  circle->mem[i * circle->dim + k] + centers->mem[j * centers->dim + k] ;

  
  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  return out ;
}


tubular_NDmesh * NDclosed_combinatoric_sum ( NDorbit *circle , NDorbit *centers )
{
  if ( circle->dim != centers->dim ){
    printf("ERROR: You are trying to create a mesh from structures with different dimensions.\n" );
    return NULL ;
  }

  unsigned triside = 3 ; /* the number of sides in a triangle! */

  tubular_NDmesh *out = tubular_closed_NDmesh_alloc( centers->number , circle->number , circle->dim );  

  unsigned long i , j ;
  unsigned k ;
  
/*   The next instruction, that is the body of this function, follows 
 *   exactly the same spirit of the similar instruction in function 
 *   NDorbit_combinatoric_sum. Please read it carefully. 
 *
 *   In the next few lines we fill up the coordinates of the mesh.
 */
  for ( j = 0 ; j < centers->number ; j++ )
    for ( i = 0 ; i < circle->number ; i++ )
      for ( k = 0 ; k < circle->dim ; k++ )
	out->coord[( i + j * circle->number ) * circle->dim + k ] =
	  circle->mem[i * circle->dim + k] + centers->mem[j * centers->dim + k] ;

  
  /* now we will define the triangles  */
  for ( k = 0 ; k < out->rad ; k++ )
    for ( j = 0 ; j < out->circ - 1 ; j++ )
      {
	if ( k < out->rad - 1 )
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 + (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
	else 
	  {
	    out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 1] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 2] = k + (j + 1) * out->rad ;
		    
	    out->tri[2 * triside * ( k + j * out->rad) + 3] = j * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 4] = (j + 1) * out->rad ;
	    out->tri[2 * triside * ( k + j * out->rad) + 5] = k + (j + 1) * out->rad ;
	  }
      }
  
  /* now we will add more triangles to close the mesh */
  j = out->circ - 1 ;
  
  for ( k = 0 ; k < out->rad ; k++ )
    {
      if ( k < out->rad - 1 ) 
	{
	  out->tri[2 * triside * ( k + j * out->rad)] = k + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 1] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 2] = k  ;
	  
	  out->tri[2 * triside * ( k + j * out->rad) + 3] = k + 1 + j * out->rad ;
	  out->tri[2 * triside * ( k + j * out->rad) + 4] = k + 1 ;
	  out->tri[2 * triside * ( k + j * out->rad) + 5] = k ;
	}
    }

  return out ;
}


void povray_mesh2 ( FILE *outfile , tubular_NDmesh *mesh )
{

  if ( outfile == NULL || mesh == NULL )
    return ;

  unsigned long i ;
  unsigned triside = 3 ;

  fprintf( outfile , 
	   "mesh2 {\n"
	   "   vertex_vectors {\n"
	   "   %ld,\n",
	   mesh->nvert ) ;
  
  
  for ( i = 0 ; i < mesh->nvert ; i++ ){
    if ( i < mesh->nvert - 1 )
      fprintf( outfile ,
	       "    < %f , %f , %f >,\n",
	       mesh->coord[i * triside ] ,   
	       mesh->coord[i * triside + 1] ,   
	       mesh->coord[i * triside + 2] ) ;
    else 
      fprintf( outfile ,
	       "    < %f , %f , %f >\n",
	       mesh->coord[i * triside ] ,   
	       mesh->coord[i * triside + 1] ,   
	       mesh->coord[i * triside + 2] ) ;
  }

  fprintf( outfile , 
	   " }\n  face_indices {\n"
	   "   %ld,\n",
	   mesh->ntri ) ;

  for ( i = 0 ; i < mesh->ntri ; i++ ){
    if ( i < mesh->ntri - 1 )
      fprintf( outfile ,
	       "    < %ld , %ld , %ld >,\n",
	       mesh->tri[i * triside ] ,   
	       mesh->tri[i * triside + 1] ,   
	       mesh->tri[i * triside + 2] ) ;
    else 
      fprintf( outfile ,
	       "    < %ld , %ld , %ld >\n",
	       mesh->tri[i * triside ] ,   
	       mesh->tri[i * triside + 1] ,   
	       mesh->tri[i * triside + 2] ) ;
  }

  fprintf( outfile ,
	   " }\n"
	   "texture {T_Wood1 scale 2} \n}" );

}

/* other textures */
/* {T_Wood1} */


/* this is a local function */
void povray_Definitions ( FILE *out ){
  fprintf(out, 
	  "global_settings { ambient_light rgb<1,1,1> }\n"
      	  );
}


int main ( int argc , char **argv )
{
  
  /* the dimension of the problem */ 
  unsigned dimension = 3 ;
    
  /* the radius around the centers */
  unsigned long nradius = 80 ;
  NDorbit *arus = NDorbit_alloc( nradius , dimension ) ;
  
  arus->mem[0] = 4 ; /* the radius of the tube */
  rotation_into_NDorbit ( arus ,2* pi/(nradius+0.0) , 0 , 1 ) ;

  /* the centers of the tube that we want to copute the mesh */
  unsigned long ncenters = 100 ;
  NDorbit *centers = NDorbit_alloc ( ncenters , dimension ) ;
  
  unsigned i ;
  for ( i = 0 ; i < ncenters ; i++ ){
    centers->mem[i*dimension + 0] = 10 * cos((2 * pi * i)/(ncenters+0.0)) ;
    centers->mem[i*dimension + 2] = 10 * sin((2 * pi * i)/(ncenters+0.0)) ;
  }



  /* Now we create a grid */
  tubular_NDmesh *tube = NDclosed_combinatoric_sum( arus , centers ) ;

  NDorbit_free(arus) ;
  NDorbit_free(centers) ;

  /* 
   * Now we will just add things to our scene.
   */


  FILE *povray_file ;
  povray_file = fopen("teste.pov" , "w" ) ;

  /* begin povray definitions */
  povray_Include( povray_file, "colors.inc" );
  povray_Include( povray_file, "functions.inc" );
  povray_Include( povray_file, "stars.inc" );
  povray_Include( povray_file, "textures.inc" );
  povray_Include( povray_file, "woods.inc" );
  povray_Include( povray_file, "metals.inc" );
  povray_Definitions(povray_file) ;

/*    double scale[3] = { 2 , 1 , 1 } ; */
/*    povray_Sky( povray_file , scale ) ; */
  
/*  char *texture = "T_Wood30 scale .5" ;*/

  TDvector position = {0,20,-40} ;
  TDvector look = {0,-10,20} ;
  TDvector rotation = {0,0,0} ;
  povray_add_Camera( povray_file , position , look , rotation ) ;
  /* end povray definitions */

  povray_mesh2 ( povray_file , tube ) ; 
  

  /* just some povray to complete the scene */
  Color testea = {.76 , .62 , .34 } ;
  Color testeb = {.99 , .99 , .80 } ;
  Color testec = {1 , 1 , 1 } ;
  
  TDvector tdir = {34,29,-16} ;
  TDvector tdi = {14,-10,-3} ;
  TDvector td = {0,1000,0} ;

  povray_add_AreaLS( povray_file , tdir , testea ) ;
  povray_add_AreaLS( povray_file , tdi  , testeb ) ;
  povray_add_AreaLS( povray_file , td  , testec ) ;

  fprintf( povray_file ,
	   "light_source\n"
	   "   { <0,0,0> color 1\n"
	   "     looks_like\n"
	   "       { sphere\n"
	   "         { <0,0,0>,10\n"
	   "           color  rgb 1 \n"
	   "           finish { ambient 1 }\n"
	   "         }\n"
	   "       }\n"
	   "     translate <10,20,30>\n"
	   "     }\n") ;

  fclose( povray_file ) ;

  
  tubular_NDmesh_free(tube) ;
  return 0 ;

}
