/*
 * AUTHOR: Gonçalo Morais (gnrm@fct.unl.pt) 
 *
 * Version :: 0.0.07 
 * STATE :: Highly Unstable.
 *
 * NUMDE is a scientific
 * library and pretends to implement the state-of-art of the
 * algorithms related with Differential Equations.  Although this
 * project is mainly written by one person, it received the valuable
 * contribution of several people. See the home page of the project
 * for further details.
 */

/*
 *  This file is part of NUMDE.
 *
 *  NUMDE is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  NUMDE is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with NUMDE.  If not, see <http://www.gnu.org/licenses/>.
 *
 */


#ifndef _NUMDE_H__
#define _NUMDE_H__

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>
#include <string.h>


/* 
 * The datatype rgb is used to produce the output for some specific
 * formats like POVRAY.
*/
typedef struct {
  double r ;
  double g ;
  double b ;
} Color ;

/* 
 * The datatype TDvector is used to position cameras and other vector
 * calculations of cameras in POVRAY.
*/
typedef struct{
  double x;
  double y;
  double z;
} TDvector;

/* For several kind of problems we need to deal with multidimensional
   structures, like in the case of N-body problem. This is the basic
   structure for that case. */

typedef struct {
  unsigned dim ;
  unsigned long int number ;
  double *mem ;
} NDorbit ;

/* the next typedef just creates a prototype for what is called a
   vector field. the arguments are two arrays of doubles, and an
   unsigned int. the first array is the place where the result will be
   printed, while the unsigned integer in the second argument,
   controls the dimension of the second array, that is the place where
   the vector field will be evaluated. */
typedef  void (*vfield)( double * , unsigned , const double *) ;


typedef struct {
  unsigned dim ;                /* the dimension of the space where mesh lives */
  unsigned rad ;                /* the number of points in each section */
  unsigned long circ ;          /* the number of sections (circles) */
  unsigned long nvert ;         /* the number of vertices */
  double *coord ;               /* where the coordinates of all this will be */
  unsigned long ntri ;          /* the number of triangles */
  unsigned long *tri ;          /* the indices of the vertex that belong to each triangle */
} tubular_NDmesh ;

typedef struct {
  unsigned dim ;                /* the dimension of the space where mesh lives */
  unsigned rad ;                /* the number of points in each section */
  unsigned long circ ;          /* the number of sections (circles) */
  unsigned long nvert ;         /* the number of vertices */
  double *coord ;               /* where the coordinates of all this will be */
  unsigned long ntri ;          /* the number of triangles */
  unsigned long *tri ;          /* the indices of the vertex that belong to each triangle */
  unsigned long nnvec ;         /* the number of normal vectors */
  double *nvec ;                /* the coordinates of normal vectors */ 
} stubular_NDmesh ;

/* Well, we need even more memory */
NDorbit* NDorbit_alloc ( unsigned long int nelements , unsigned int dimension ) ;
void NDorbit_free ( NDorbit *in ) ;

void NDorbit_print ( NDorbit *in ) ;
NDorbit* NDorbit_diff ( NDorbit *in ) ;
NDorbit* NDorbit_combinatoric_sum ( NDorbit *out , NDorbit *in ) ;
NDorbit* NDorbit_cut ( NDorbit *in , unsigned gap ) ;

/* things about meshes */
tubular_NDmesh* tubular_NDmesh_alloc ( unsigned long ncirc , unsigned nrad , unsigned dimension ) ;
tubular_NDmesh* tubular_closed_NDmesh_alloc ( unsigned long ncirc , unsigned nrad , unsigned dimension ) ;
stubular_NDmesh* stubular_closed_NDmesh_alloc ( unsigned long ncirc , unsigned nrad , unsigned dimension ) ;
stubular_NDmesh* stubular_NDmesh_alloc ( unsigned long ncirc , unsigned nrad , unsigned dimension ) ;

/* serves perfectly for open and closed mesh's */
void tubular_NDmesh_free ( tubular_NDmesh *in ) ;
void stubular_NDmesh_free ( stubular_NDmesh *in ) ;

void tubular_NDmesh_coord_print ( tubular_NDmesh *mesh ) ;
void tubular_NDmesh_tri_print ( tubular_NDmesh *mesh ) ;

tubular_NDmesh * NDcombinatoric_sum ( NDorbit *circle , NDorbit *centers ) ;
tubular_NDmesh * NDclosed_combinatoric_sum ( NDorbit *circle , NDorbit *centers ) ;
stubular_NDmesh * NDscombinatoric_sum ( NDorbit *circle , NDorbit *centers ) ;
stubular_NDmesh* NDclosed_scombinatoric_sum ( NDorbit *circle , NDorbit *centers ) ;


/* Now a giant step for men, a small step for the mankind */
void ND_RungeKutta ( NDorbit *out , unsigned long position , double tj , double *kk , vfield function) ;
NDorbit* run_RungeKutta ( unsigned long int nsteps , unsigned dimension , double *in , double tj , vfield function) ;

/* The prototypes of vfields used in simulations. */
void Lorenz ( double *out , unsigned dimension , const double *in , double a , double b , double c ) ;
void CLorenz ( double *out , unsigned dimension , const double *in ) ;
void Rossler ( double *out , unsigned dimension , const double *in ) ;  

/* vector calculus */
double Dot ( double *va , double *vb , unsigned dimension ) ;
void Normalization( double *vector , unsigned dimension ) ;
double* Projection( double *vector , unsigned dimension , unsigned ortdir ) ;
void EulerAngles ( const NDorbit *radius , double *vector , NDorbit *out ) ;

/* Unstable part */
stubular_NDmesh * NDTestSum ( NDorbit *centers , NDorbit *radius , void (*vectorf)(double * , double * )) ;
stubular_NDmesh* NDTestSumP ( NDorbit *centers , 
			      NDorbit *radius, 
			      double *parameter ,
			      void (*vectorf)(double * , double )) ;
void NDmeshCoordPrintST ( stubular_NDmesh *mesh ) ;

#endif
